<?php

namespace Toogas\Easypay\Controller\Gateway;

use Magento\Framework\App\Action\Context;
use Magento\Framework\App\Config\ScopeConfigInterface;

class BoletoBancario extends \Magento\Framework\App\Action\Action
{
    const EASYPAY_METHOD = 'bb';

    const API_URL                                       = 'https://www.easypay.pt/_s/';
    const DEBUG_URL                                     = 'http://test.easypay.pt/_s/';

    /** @var \Magento\Checkout\Model\Session */
    protected $_checkoutSession;

    /** @var \Toogas\Easypay\Model\Http\EasypayApi */
    protected $_easypayApi;

    /** @var \Toogas\Easypay\Model\Config\EasypayConfig */
    protected $_easypayConfig;

    /** @var \Magento\Sales\Model\OrderRepository */
    protected $_orderRepository;

    /** @var \Magento\Sales\Model\OrderFactory */
    protected $_orderFactory;

    /** @var \Magento\Sales\Model\Service\InvoiceService */
    protected $_invoiceService;

    /** @var \Magento\Framework\DB\Transaction */
    protected $_transaction;

    /** @var ScopeConfigInterface */
    protected $scopeConfig;

    /** @var \Toogas\Easypay\Model\EasypayPaymentFactory */
    protected $_easypayPaymentFactory;

    /** @var \Toogas\Easypay\Model\EasypayApiTransactionFactory */
    protected $_easypayApiTransactionFactory;

    /**
     * BoletoBancario constructor.
     * @param Context $context
     * @param \Magento\Checkout\Model\Session $checkoutSession
     * @param \Toogas\Easypay\Model\Http\EasypayApi $easypayApi
     * @param \Toogas\Easypay\Model\Config\EasypayConfig $easypayConfig
     * @param \Magento\Sales\Model\OrderRepository $orderRepository
     * @param \Magento\Sales\Model\OrderFactory $orderFactory
     * @param \Magento\Sales\Model\Service\InvoiceService $invoiceService
     * @param \Magento\Framework\DB\Transaction $transaction
     * @param ScopeConfigInterface $scopeConfig
     * @param \Toogas\Easypay\Model\EasypayPaymentFactory $easypayPaymentFactory
     * @param \Toogas\Easypay\Model\EasypayApiTransactionFactory $easypayApiTransactionFactory
     */
    public function __construct(
        Context $context,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Toogas\Easypay\Model\Http\EasypayApi $easypayApi,
        \Toogas\Easypay\Model\Config\EasypayConfig $easypayConfig,
        \Magento\Sales\Model\OrderRepository $orderRepository,
        \Magento\Sales\Model\OrderFactory $orderFactory,
        \Magento\Sales\Model\Service\InvoiceService $invoiceService,
        \Magento\Framework\DB\Transaction $transaction,
        ScopeConfigInterface $scopeConfig,
        \Toogas\Easypay\Model\EasypayPaymentFactory $easypayPaymentFactory,
        \Toogas\Easypay\Model\EasypayApiTransactionFactory $easypayApiTransactionFactory
    ) {
        parent::__construct($context);
        $this->_checkoutSession = $checkoutSession;
        $this->_easypayApi = $easypayApi;
        $this->_easypayConfig = $easypayConfig;
        $this->_orderRepository = $orderRepository;
        $this->_orderFactory = $orderFactory;
        $this->_invoiceService = $invoiceService;
        $this->_transaction = $transaction;
        $this->scopeConfig = $scopeConfig;
        $this->_easypayPaymentFactory = $easypayPaymentFactory;
        $this->_easypayApiTransactionFactory = $easypayApiTransactionFactory;
    }

    /**
     * @return \Magento\Framework\App\ResponseInterface|\Magento\Framework\Controller\ResultInterface
     * @throws \Exception
     */
    public function execute()
    {
        $status = $this->_request->getParam('s');
        if ($status !== null) {
            $params = $this->_request->getParams();
            return $this->handleForward($params);
        }
        $order = $this->_checkoutSession->getLastRealOrder();
        $data = $this->generateBoletoRefs($order);
        if ($data && is_array($data) && isset($data['ep_boleto']) && is_string($data['ep_boleto']) && !empty($data['ep_boleto'])) {
            $reditectUrl = $data['ep_boleto'];
            $order->getPayment()->setAdditionalInformation('easypay', $data);
            $order->save();
            /** @var \Toogas\Easypay\Model\EasypayPayment $payment */
            $payment = $this->_easypayPaymentFactory->create();
            $payment->setData([
                'order_id'      => $order->getIncrementId(),
                'payment_id'    => $data['ep_entity'] . $data['ep_reference'],
                'status'        => 'created',
                'type'          => 'authorisation',
                'method'        => self::EASYPAY_METHOD
            ]);
            $payment->save();
            /** @var \Toogas\Easypay\Model\EasypayApiTransaction $transaction */
            $transaction = $this->_easypayApiTransactionFactory->create();
            $transaction->setData([
                'payment_id'            => $data['ep_entity'] . $data['ep_reference'],
                'origin'                => 'magento',
                'api_transaction_json'  => json_encode($data, JSON_PRETTY_PRINT)
            ]);
            $transaction->save();
            return $this->resultRedirectFactory->create()->setUrl($reditectUrl);
        }
        return $this->_redirect('easypay/checkout/failed');
    }

    /**
     * @param $params
     * @return \Magento\Framework\App\ResponseInterface
     * @throws \Exception
     */
    protected function handleForward($params)
    {
        if (isset($params['e']) && $params['e'] !== '' && isset($params['r']) && $params['r'] !== '') {
            $orderModel = $this->_orderFactory->create();
            $payment_id = $params['e'] . $params['r'];
            $order = $orderModel->loadByIncrementId($params['t_key']);
            if ($order && $order->getId()) {
                $payment_type = $this->getPaymentType();
                $order->getPayment()->setAdditionalInformation('easypay_forward', $params);
                if ($params['s'] === 'ok') {
                    /** @var \Toogas\Easypay\Model\EasypayPayment $payment */
                    $payment = $this->_easypayPaymentFactory->create();
                    $payment->load($payment_id, 'payment_id');
                    $payment->setStatus('authorized');
                    $payment->save();
                    /** @var \Toogas\Easypay\Model\EasypayApiTransaction $transaction */
                    $transaction = $this->_easypayApiTransactionFactory->create();
                    $transaction->setData([
                        'payment_id'            => $payment_id,
                        'origin'                => 'easypay',
                        'api_transaction_json'  => json_encode($params, JSON_PRETTY_PRINT)
                    ]);
                    $transaction->save();
                    $order->setState(\Magento\Sales\Model\Order::STATE_PENDING_PAYMENT);
                    $order->setStatus(\Magento\Sales\Model\Order::STATE_PENDING_PAYMENT);
                    $order->save();
                    return $this->_redirect('checkout/onepage/success');
                }
            }
        }
        $order->cancel();
        $order->save();
        return $this->_redirect('easypay/checkout/failed');
    }

    /**
     * @return mixed
     */
    protected function getPaymentType()
    {
        return $this->_easypayConfig->getCreditCard()['payment_type'];
    }

    /**
     * Generate Boleto references
     * @param \Magento\Sales\Model\Order $order
     * @return boolean
     */
    public function generateBoletoRefs($order)
    {
        if ($order) {
            $params = $this->getRequestParams($order);
            $request = $this->getEasyPayUrl() . 'api_easypay_01BG.php?' . implode('&', $params);
            return $this->xmlToArray($this->get($request));
        }
        return false;
    }

    /**
     * @param $url
     * @return bool|string
     */
    private function get($url)
    {
        $ch = curl_init();

        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        $output = curl_exec($ch);

        curl_close($ch);
        return $output;
    }

    private function getRequestParams($order)
    {
        $params = [];
        $params['ep_entity'] = $this->getV1ApiConfig('entidade');
        $params['ep_cin'] = $this->getV1ApiConfig('cin');
        $params['ep_user'] = $this->getV1ApiConfig('userid');
        $params['ep_ref_type'] = 'auto';
        $params['ep_type'] = 'boleto';
        //$params['ep_currency'] = 'BRL';
        $params['ep_country'] = 'PT';
        $params['ep_language'] = 'PT';
        $params['t_value'] = $order->getGrandTotal();
        $params['t_key'] = $order->getIncrementId();
        $params['o_name'] = $order->getCustomerName();
        $params['o_email'] = $order->getCustomerEmail();
        $params['s_code'] = $this->getV1ApiConfig('s_code');
        if ($this->isSandbox()) {
            //$params[] = 'ep_partner=' . $this->getV1ApiConfig('userid');
            //$params[] = 'o_max_date=' . $this->getOMAxDate();
        }
        $url_params = [];
        foreach ($params as $key => $value) {
            $url_params[] = $key . '=' . $value;
        }
        return $url_params;
    }

    /**
     * @return string
     */
    private function getEasyPayUrl()
    {
        if ($this->isSandbox()) {
            return self::DEBUG_URL;
        }
        return self::API_URL;
    }

    /**
     * @param $key
     * @return mixed
     */
    private function getV1ApiConfig($key)
    {
        return $this->scopeConfig->getValue('payment/easypay_boleto/' . $key);
    }

    /**
     * @return string|null
     */
    public function getOMAxDate()
    {
        return date('Y-m-d');
    }

    private function isSandbox()
    {
        return ((string) $this->scopeConfig->getValue('payment/easypay_configure/sandbox') === '1');
    }

    private function xmlToArray($xml)
    {
        $obj = @simplexml_load_string($xml);
        if (!$obj) {
            return false;
        }
        return $this->objectToArray($obj);
    }

    private function objectToArray($obj)
    {
        if (!is_object($obj) && !is_array($obj)) {
            return $obj;
        }
        if (is_object($obj)) {
            $obj = get_object_vars($obj);
        }
        return array_map([$this, 'objectToArray'], $obj);
    }



}
