<?php

namespace Toogas\Easypay\Controller\Gateway;

use Magento\Framework\App\Action\Context;

class CreditCard extends \Magento\Framework\App\Action\Action
{
    const EASYPAY_METHOD = 'cc';

    /** @var \Magento\Checkout\Model\Session */
    protected $_checkoutSession;

    /** @var \Toogas\Easypay\Model\Http\EasypayApi */
    protected $_easypayApi;

    /** @var \Toogas\Easypay\Model\Config\EasypayConfig */
    protected $_easypayConfig;

    /** @var \Magento\Sales\Model\OrderRepository */
    protected $_orderRepository;

    /** @var \Magento\Sales\Model\OrderFactory */
    protected $_orderFactory;

    /** @var \Magento\Sales\Model\Service\InvoiceService */
    protected $_invoiceService;

    /** @var \Magento\Framework\DB\Transaction */
    protected $_transaction;

    /** @var \Toogas\Easypay\Model\EasypayPaymentFactory */
    protected $_easypayPaymentFactory;

    /** @var \Toogas\Easypay\Model\EasypayApiTransactionFactory */
    protected $_easypayApiTransactionFactory;

    public function __construct(
        Context $context,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Toogas\Easypay\Model\Http\EasypayApi $easypayApi,
        \Toogas\Easypay\Model\Config\EasypayConfig $easypayConfig,
        \Magento\Sales\Model\OrderRepository $orderRepository,
        \Magento\Sales\Model\OrderFactory $orderFactory,
        \Magento\Sales\Model\Service\InvoiceService $invoiceService,
        \Magento\Framework\DB\Transaction $transaction,
        \Toogas\Easypay\Model\EasypayPaymentFactory $easypayPaymentFactory,
        \Toogas\Easypay\Model\EasypayApiTransactionFactory $easypayApiTransactionFactory
    ) {
        parent::__construct($context);
        $this->_checkoutSession = $checkoutSession;
        $this->_easypayApi = $easypayApi;
        $this->_easypayConfig = $easypayConfig;
        $this->_orderRepository = $orderRepository;
        $this->_orderFactory = $orderFactory;
        $this->_invoiceService = $invoiceService;
        $this->_transaction = $transaction;
        $this->_easypayPaymentFactory = $easypayPaymentFactory;
        $this->_easypayApiTransactionFactory = $easypayApiTransactionFactory;
    }

    public function execute()
    {
        $status = $this->_request->getParam('s');
        if ($status !== null) {
            $params = $this->_request->getParams();
            return $this->handleForward($params);
        }
        $order = $this->_checkoutSession->getLastRealOrder();
        $frequent = $this->_request->getParam('f');
        $easypay_method = self::EASYPAY_METHOD;
        if ($frequent !== null) {
            $easypay_method = 'fq-' . $easypay_method;
        }
        $response = $this->sendPaymentRequest($order, $frequent);
        if ($response&& isset($response['status'], $response['method']['url']) && $response['status'] === 'ok') {
            $redirectUrl = $response['method']['url'];
            $order->getPayment()->setAdditionalInformation('easypay', $response);
            $order->save();
            /** @var \Toogas\Easypay\Model\EasypayPayment $payment */
            $payment = $this->_easypayPaymentFactory->create();
            $payment->setData([
                'order_id'      => $order->getIncrementId(),
                'payment_id'    => $response['id'],
                'status'        => 'created',
                'type'          => $this->getPaymentType(),
                'method'        => $easypay_method
            ]);
            $payment->save();
            /** @var \Toogas\Easypay\Model\EasypayApiTransaction $transaction */
            $transaction = $this->_easypayApiTransactionFactory->create();
            $transaction->setData([
                'payment_id'            => $response['id'],
                'origin'                => 'magento',
                'api_transaction_json'  => json_encode($response, JSON_PRETTY_PRINT)
            ]);
            $transaction->save();
            return $this->resultRedirectFactory->create()->setUrl($redirectUrl);
        }
        $order->cancel();
        $order->save();
        return $this->_redirect('easypay/checkout/failed');
    }

    protected function handleForward($params)
    {
        if (isset($params['s'])) {
            if (isset($params['t_key']) && $params['t_key'] !== '') {
                $orderModel = $this->_orderFactory->create();
                $order = $orderModel->loadByIncrementId($params['t_key']);
                if ($order && $order->getId()) {
                    $payment_type = $this->getPaymentType();
                    $order->getPayment()->setAdditionalInformation('easypay_forward', $params);
                    /**
                    if ($params['s'] === 'ok') {
                        return $this->_redirect('checkout/onepage/success');
                    }
                    $order->cancel();
                    $order->save();
                    $this->messageManager->addErrorMessage(__('Payment Failed, the order was cancelled.'));
                    return $this->_redirect('easypay/checkout/failed');
                     */
                }
            }
        }
        return $this->_redirect('checkout/onepage/success');
    }

    protected function getPaymentType()
    {
        return $this->_easypayConfig->getCreditCard()['payment_type'];
    }

    protected function sendPaymentRequest(\Magento\Sales\Model\Order $order, $frequent=false)
    {
        if ($frequent !== false && $frequent !== null) {
            return $this->sendFrequentRequest($order);
        }
        return $this->sendSingleRequest($order);
    }

    protected function sendFrequentRequest(\Magento\Sales\Model\Order $order)
    {
        $frequentConfig = $this->_easypayConfig->getFrequent();
        $body = [
            'key' => $order->getIncrementId(),
            'currency' => $order->getBaseCurrency()->getCode(),
            'max_value' => (float) $frequentConfig['max_value'],
            'min_value' => (float) $frequentConfig['min_value'],
            'unlimited_payments' => true,
        ];
        $billingInfo = $order->getBillingAddress();
        $body['customer'] = [
            'key' => $order->getCustomerId() ?? '',
            'name' => $order->getCustomerName() ?? '',
            'email' => $order->getCustomerEmail() ?? '',
            'phone' => $billingInfo->getTelephone() ?? '',
            'fiscal_number' => $order->getCustomerTaxvat() ?? '',
        ];
        return $this->_easypayApi->createFrequentPayment(
            'cc',
            $body
        );
    }

    protected function sendSingleRequest(\Magento\Sales\Model\Order $order)
    {
        $body = [
            'key' => $order->getIncrementId()
        ];
        $billingInfo = $order->getBillingAddress();
        $body['customer'] = [
            'key' => $order->getCustomerId() ?: '',
            'name' => $order->getCustomerName() ?: '',
            'email' => $order->getCustomerEmail() ?: '',
            'phone' => $billingInfo->getTelephone() ?: '',
            'fiscal_number' => $order->getCustomerTaxvat() ?: '',
        ];
        return $this->_easypayApi->createSinglePayment(
            self::EASYPAY_METHOD,
            (float) $order->getGrandTotal(),
            $this->getPaymentType(),
            'Pagamento de Encomenda',
            $body
        );
    }
}
