<?php

namespace Toogas\Easypay\Model\Http;

class EasypayApi
{
    const API_URL                                       = 'https://api.prod.easypay.pt/2.0/';
    const DEBUG_URL                                     = 'https://api.test.easypay.pt/2.0/';

    const REQUEST_CONTENT_TYPE                          = 'application/json';
    const REQUEST_SINGLE_PAYMENT                        = 'single';
    const REQUEST_FREQUENT_PAYMENT                      = 'frequent';
    const REQUEST_PAYMENT_VOID                          = 'void';
    const REQUEST_PAYMENT_CAPTURE                       = 'capture';
    const REQUEST_PAYMENT_AUTH_DETAILS                  = 'authorisation';

    const STATUS_CODE_SUCCESS                           = 'ok';
    const STATUS_CODE_ERROR                             = 'error';

    const FREQ_CALLBACK_URL                             = '/easypay/frequent';

    /** @var \Toogas\Easypay\Helper\Http */
    protected $_httpRequestHelper;

    /** @var \Toogas\Easypay\Model\Config\EasypayConfig */
    protected $_easypayConfig;

    /** @var string */
    protected $_configuration;

    /**
     * EasypayApi constructor.
     * @param \Toogas\Easypay\Helper\Http $httpRequestHelper
     * @param \Toogas\Easypay\Model\Config\EasypayConfig $easypayConfig
     */
    public function __construct(
        \Toogas\Easypay\Helper\Http $httpRequestHelper,
        \Toogas\Easypay\Model\Config\EasypayConfig $easypayConfig
    ) {
        $this->_httpRequestHelper = $httpRequestHelper;
        $this->_easypayConfig = $easypayConfig;
    }

    /**
     * Create Single Payment - https://api.test.easypay.pt/docs#tag/Single-Payment
     * @param string $method
     * @param float $value
     * @param string $type
     * @param string $descriptive
     * @param array $body
     * @return array|bool|string
     */
    public function createSinglePayment($method, $value, $type='sale', $descriptive='Pagamento de Encomenda', $body = [])
    {
        $body['method'] = $method;
        $body['value'] = $value;
        $body['type'] = $type;
        $body['capture'] = ['descriptive' => $descriptive];
        $request_url = $this->getApiUrl() . self::REQUEST_SINGLE_PAYMENT;
        return $this->requestApi($request_url, $body);
    }

    /**
     * Show Single Payment - https://api.test.easypay.pt/docs#tag/Single-Payment
     * @param $id
     * @return array|bool|string
     */
    public function showSinglePayment($id)
    {
        $request_url = $this->getApiUrl() . self::REQUEST_SINGLE_PAYMENT . '/' . $id;
        return $this->requestApi($request_url, [], 'GET');
    }

    /**
     * List Single Payment - https://api.test.easypay.pt/docs#tag/Single-Payment
     * @param int $page
     * @param int $records_per_page
     * @return array|bool|string
     */
    public function listSinglePayment($page=1, $records_per_page=100)
    {
        $request_url = $this->getApiUrl() . self::REQUEST_SINGLE_PAYMENT;
        return $this->requestApi($request_url, ['page' => $page, 'records_per_page' => $records_per_page], 'GET');
    }

    /**
     * Delete Single Payment - https://api.test.easypay.pt/docs#tag/Single-Payment
     * @param $id
     * @return array|bool|string
     */
    public function deleteSinglePayment($id)
    {
        $request_url = $this->getApiUrl() . self::REQUEST_SINGLE_PAYMENT . '/' . $id;
        return $this->requestApi($request_url, [], 'DELETE');
    }

    /**
     * Delete Single Payment - https://api.test.easypay.pt/docs#tag/Single-Payment
     * @param $id
     * @param $body
     * @return array|bool|string
     */
    public function updateSinglePayment($id, $body)
    {
        $request_url = $this->getApiUrl() . self::REQUEST_SINGLE_PAYMENT . '/' . $id;
        return $this->requestApi($request_url, $body, 'PATCH');
    }

    /**
     * Create Frequent Payment - https://api.test.easypay.pt/docs#tag/Frequent-Payment
     * @param $method
     * @param array $body
     * @return array|bool|string
     */
    public function createFrequentPayment($method, $body=[])
    {
        $body['method'] = $method;
        $request_url = $this->getApiUrl() . self::REQUEST_FREQUENT_PAYMENT;
        return $this->requestApi($request_url, $body);
    }

    /**
     * List Frequent Payment - https://api.test.easypay.pt/docs#tag/Frequent-Payment
     * @param int $page
     * @param int $records_per_page
     * @return array|bool|string
     */
    public function listFrequentPayment($page=1, $records_per_page=100)
    {
        $request_url = $this->getApiUrl() . self::REQUEST_FREQUENT_PAYMENT;
        return $this->requestApi($request_url, ['page' => $page, 'records_per_page' => $records_per_page], 'GET');
    }

    /**
     * Show Frequent Payment - https://api.test.easypay.pt/docs#tag/Frequent-Payment
     * @param $id
     * @return array|bool|string
     */
    public function showFrequentPayment($id)
    {
        $request_url = $this->getApiUrl() . self::REQUEST_FREQUENT_PAYMENT . '/' . $id;
        return $this->requestApi($request_url, [], 'GET');
    }

    /**
     * Delete Frequent Payment - https://api.test.easypay.pt/docs#tag/Frequent-Payment
     * @param $id
     * @return array|bool|string
     */
    public function deleteFrequentPayment($id)
    {
        $request_url = $this->getApiUrl() . self::REQUEST_FREQUENT_PAYMENT . '/' . $id;
        return $this->requestApi($request_url, [], 'DELETE');
    }

    /**
     * Update Frequent Payment - https://api.test.easypay.pt/docs#tag/Frequent-Payment
     * @param $id
     * @param $body
     * @return array|bool|string
     */
    public function updateFrequentPayment($id, $body)
    {
        $request_url = $this->getApiUrl() . self::REQUEST_FREQUENT_PAYMENT . '/' . $id;
        return $this->requestApi($request_url, $body, 'PATCH');
    }

    /**
     * Request Frequent Payment Authorization - https://api.test.easypay.pt/docs#tag/Frequent-Payment
     * @param $id
     * @param $value
     * @param string $descriptive
     * @param array $body
     * @return array|bool|string
     */
    public function requestFrequentPaymentAuth($id, $value, $descriptive='Pagamento de Subscrição', $body=[])
    {
        $body['value'] = (float) $value;
        $body['descriptive'] = $descriptive;
        $request_url = $this->getApiUrl() . self::REQUEST_FREQUENT_PAYMENT . '/authorisation/' . $id;
        return $this->requestApi($request_url, $body);
    }

    /**
     * Request Payment Capture - https://api.test.easypay.pt/docs#tag/Payment-Generic-Operations
     * @param $id
     * @param $value
     * @param string $descriptive
     * @param $body
     * @return array|bool|string
     */
    public function requestPaymentCapture($id, $value, $descriptive='Pagamento de Encomenda', $body=[])
    {
        $body['value'] = (float) $value;
        $body['descriptive'] = $descriptive;
        $request_url = $this->getApiUrl() . self::REQUEST_PAYMENT_CAPTURE . '/' . $id;
        return $this->requestApi($request_url, $body);
    }

    /**
     * Request Payment Authorization Details - https://api.test.easypay.pt/docs#tag/Payment-Generic-Operations
     * @param $id
     * @return array|bool|string
     */
    public function requestPaymentAuthDetails($id)
    {
        $request_url = $this->getApiUrl() . self::REQUEST_PAYMENT_AUTH_DETAILS . '/' . $id;
        return $this->requestApi($request_url, [], 'GET');
    }

    /**
     * Request Payment Void - https://api.test.easypay.pt/docs#tag/Payment-Generic-Operations
     * @param $id
     * @param string $descriptive
     * @param $body
     * @return array|bool|string
     */
    public function requestPaymentVoid($id, $descriptive='Cancelamento de Encomenda', $body=[])
    {
        $body['descriptive'] = $descriptive;
        $request_url = $this->getApiUrl() . self::REQUEST_PAYMENT_VOID . '/' . $id;
        return $this->requestApi($request_url, $body);
    }

    /**
     * Handle Generic Notification JSON - https://api.prod.easypay.pt/docs#tag/Notification
     * @param $json
     */
    public function handleGenericNotification($json)
    {
        $notification = json_decode($json);
        if ($notification === null || !isset($notification['id'])) {
            return;
        }
    }

    /**
     * Handle Authorization Notification JSON - https://api.prod.easypay.pt/docs#tag/Notification
     * @param $json
     */
    public function handleAuthorizationNotification($json)
    {
        $notification = json_decode($json);
        if ($notification === null || !isset($notification['id'])) {
            return;
        }
    }

    /**
     * Handle Transaction Notification JSON - https://api.prod.easypay.pt/docs#tag/Notification
     * @param $json
     */
    public function handleTransactionNotification($json)
    {
        $notification = json_decode($json);
        if ($notification === null || !isset($notification['id'])) {
            return;
        }
    }

    /**
     * Request Api
     * @param $url
     * @param string $method
     * @param array $data
     * @return array|bool
     */
    public function requestApi($url, $data=[], $method='POST')
    {
        $response = $this->_httpRequestHelper->requestCurlPostJson($method, $url, $data, $this->getApiHeaders());
        if ($response) {
            return $response;
        }
        return false;
    }

    /**
     * Get Headers for Api Request
     * @return array
     */
    protected function getApiHeaders()
    {
        $data = $this->getApiData();
        $headers = [
            'AccountId: ' . ($data['accountid'] ?? ''),
            'ApiKey: ' . ($data['apikey'] ?? ''),
            'Content-Type: ' . self::REQUEST_CONTENT_TYPE
        ];
        return $headers;
    }

    /**
     * Get Api Url (Production or Sandbox)
     * @return string
     */
    protected function getApiUrl()
    {
        if ((string) $this->getConfiguration('sandbox') === '1') {
            return self::DEBUG_URL;
        }
        return self::API_URL;
    }

    /**
     * Get Client Data
     * @return array
     */
    protected function getApiData()
    {
        $data = [
            'accountid' => (string) $this->getConfiguration('accountid'),
            'apikey' => (string) $this->getConfiguration('apikey')
        ];
        return $data;
    }

    /**
     * Get Configuration
     * @param null $code
     * @return mixed
     */
    protected function getConfiguration($code=null)
    {
        if (!isset($this->_configuration)) {
            $this->_configuration = $this->_easypayConfig->getConfigure();
        }
        return $code ? $this->_configuration[$code] : $this->_configuration;
    }
}
