<?php

namespace Toogas\Easypay\Model\Payment;

use Magento\Directory\Helper\Data as DirectoryHelper;
use Magento\Framework\HTTP\ResponseFactory;
use Magento\Framework\UrlInterface;

class EasypayFrequent extends \Magento\Payment\Model\Method\AbstractMethod
{
    const EASYPAY_METHOD = 'fq';

    /** @var \Toogas\Easypay\Model\Http\EasypayApi */
    protected $_easypayApi;

    /** @var \Toogas\Easypay\Model\Config\EasypayConfig */
    protected $_easypayConfig;

    /** @var \Toogas\Easypay\Model\EasypayPaymentFactory */
    protected $_easypayPaymentFactory;

    /** @var \Toogas\Easypay\Model\EasypayApiTransactionFactory */
    protected $_easypayApiTransactionFactory;

    /** @var \Magento\Checkout\Model\Session */
    protected $_checkoutSession;

    /** @var \Toogas\Easypay\Block\Adminhtml\Payment\Info  */
    protected $_infoBlockType = \Toogas\Easypay\Block\Adminhtml\Payment\Info::class;

    protected $_code = 'easypay_frequent';
    protected $_isInitializeNeeded = true;

    /**
     * EasypayFrequent constructor.
     * @param \Magento\Framework\Model\Context $context
     * @param \Magento\Framework\Registry $registry
     * @param \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory
     * @param \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory
     * @param \Magento\Payment\Helper\Data $paymentData
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Payment\Model\Method\Logger $logger
     * @param \Toogas\Easypay\Model\Http\EasypayApi $easypayApi
     * @param \Magento\Checkout\Model\Session $checkoutSession
     * @param \Toogas\Easypay\Model\Config\EasypayConfig $easypayConfig
     * @param \Toogas\Easypay\Model\EasypayPaymentFactory $easypayPaymentFactory
     * @param \Toogas\Easypay\Model\EasypayApiTransactionFactory $easypayApiTransactionFactory
     * @param \Magento\Framework\Model\ResourceModel\AbstractResource|null $resource
     * @param \Magento\Framework\Data\Collection\AbstractDb|null $resourceCollection
     * @param array $data
     * @param DirectoryHelper|null $directory
     */
    public function __construct(
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory,
        \Magento\Payment\Helper\Data $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Payment\Model\Method\Logger $logger,
        \Toogas\Easypay\Model\Http\EasypayApi $easypayApi,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Toogas\Easypay\Model\Config\EasypayConfig $easypayConfig,
        \Toogas\Easypay\Model\EasypayPaymentFactory $easypayPaymentFactory,
        \Toogas\Easypay\Model\EasypayApiTransactionFactory $easypayApiTransactionFactory,
        \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
        array $data = [],
        DirectoryHelper $directory = null
    ) {
        parent::__construct(
            $context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $paymentData,
            $scopeConfig,
            $logger,
            $resource,
            $resourceCollection,
            $data,
            $directory
        );
        $this->_easypayApi = $easypayApi;
        $this->_easypayConfig = $easypayConfig;
        $this->_easypayPaymentFactory = $easypayPaymentFactory;
        $this->_easypayApiTransactionFactory = $easypayApiTransactionFactory;
        $this->_checkoutSession = $checkoutSession;
    }

    /**
     * @param \Magento\Quote\Api\Data\CartInterface|null $quote
     * @return bool
     */
    public function isAvailable(\Magento\Quote\Api\Data\CartInterface $quote = null)
    {
        $availableForGuest = $this->getConfigData('availableforguests');
        if($availableForGuest == 0 && $this->_checkoutSession->getQuote()->getCustomerId() === null) {
            return false;
        }
        return $this->getConfigData('methods') ? parent::isAvailable($quote) : false;
    }

    public function validate()
    {
        $info = $this->getInfoInstance()->getAdditionalInformation('checkout_data');
        if (!isset($info['payment_method']) ||
            $info['payment_method'] === '') {
            throw new \Magento\Framework\Exception\LocalizedException(
                __('You need to choose a valid Payment Method.')
            );
        }
        if ($info['payment_method'] === 'dd') {
            if (!isset($info['account_holder']) ||
                $info['account_holder'] === '') {
                throw new \Magento\Framework\Exception\LocalizedException(
                    __('You need to input Account Holder Name.')
                );
            }
            if (!isset($info['iban']) ||
                $info['iban'] === '') {
                throw new \Magento\Framework\Exception\LocalizedException(
                    __('You need to fill in your IBAN.')
                );
            }
        }
        if ($info['payment_method'] === 'mbw') {
            if (!isset($info['phone_number'])          ||
                $info['phone_number'] === ''           ||
                !ctype_digit($info['phone_number'])    ||
                strlen($info['phone_number']) !== 9) {
                throw new \Magento\Framework\Exception\LocalizedException(
                    __('You need to input a valid Phone Number')
                );
            }
        }
        return parent::validate();
    }

    /**
     * @param string $paymentAction
     * @param object $stateObject
     * @return \Magento\Payment\Model\Method\AbstractMethod|EasypayMbway
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function initialize($paymentAction, $stateObject)
    {
        $state = \Magento\Sales\Model\Order::STATE_NEW;
        $status = 'pending';
        $stateObject->setState($state);
        $stateObject->setStatus($status);
        $stateObject->setIsNotified(false);
        $order = $this->getInfoInstance()->getOrder();
        return $this->requestFq($order);
    }

    /**
     * @param \Magento\Sales\Model\Order $order
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function requestFq($order)
    {
        $orderInfo = $this->getInfoInstance();
        $data = $orderInfo->getAdditionalInformation('checkout_data');
        $method = $data['payment_method'];
        switch ($method) {
            case 'mb':
                $this->requestMultibanco($order);
                break;
            case 'mbw':
                $this->requestMbway($order, $data);
                break;
            case 'cc':
                //$this->requestCreditCard($order);
                break;
            case 'dd':
                $this->requestDirectDebit($order, $data);
                break;
            default:
                throw new \Magento\Framework\Exception\LocalizedException(
                    __('Selected Method Does not exist.')
                );
        }
        return $this;
    }

    /**
     * @param \Magento\Sales\Model\Order $order
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    private function requestMultibanco($order)
    {
        $frequentConfig = $this->_easypayConfig->getFrequent();
        $orderInfo = $this->getInfoInstance();
        $multibancoConfig = $this->_easypayConfig->getMultibanco();
        $valid_days = 7;
        if (isset($multibancoConfig['valid_days']) && $multibancoConfig['valid_days'] > 0) {
            $valid_days = $multibancoConfig['valid_days'];
        }
        $expiration_time = date('Y-m-d H:i', strtotime('+' . $valid_days . ' days'));
        $body = [
            'key' => $order->getIncrementId(),
            'currency' => $order->getBaseCurrency()->getCode(),
            'max_value' => (float) $frequentConfig['max_value'],
            'min_value' => (float) $frequentConfig['min_value'],
            'unlimited_payments' => true,
            'expiration_time' => $expiration_time,
        ];
        $billingInfo = $order->getBillingAddress();
        $body['customer'] = [
            'key' => $order->getCustomerId() ?? '',
            'name' => $order->getCustomerName() ?? '',
            'email' => $order->getCustomerEmail() ?? '',
            'phone' => $billingInfo->getTelephone() ?? '',
            'fiscal_number' => $order->getCustomerTaxvat() ?? '',
        ];
        $response = $this->_easypayApi->createFrequentPayment(
            'mb',
            $body
        );
        if ($response) {
            if (isset($response['status']) && $response['status'] === 'ok') {
                $orderInfo->setAdditionalInformation('easypay', $response);
                $orderInfo->setAdditionalInformation('checkout_data', ['expiration_time' => $expiration_time]);
                /** @var \Toogas\Easypay\Model\EasypayPayment $payment */
                $payment = $this->_easypayPaymentFactory->create();
                $payment->setData([
                    'order_id'      => $order->getIncrementId(),
                    'payment_id'    => $response['id'],
                    'status'        => 'created',
                    'type'          => 'frequent_create',
                    'method'        => self::EASYPAY_METHOD . '-mb'
                ]);
                $payment->save();
                /** @var \Toogas\Easypay\Model\EasypayApiTransaction $transaction */
                $transaction = $this->_easypayApiTransactionFactory->create();
                $transaction->setData([
                    'payment_id'            => $response['id'],
                    'origin'                => 'magento',
                    'api_transaction_json'  => json_encode($response, JSON_PRETTY_PRINT)
                ]);
                $transaction->save();
            } else {
                throw new \Magento\Framework\Exception\LocalizedException(
                    __('Could not process your payment.')
                );
            }
        } else {
            throw new \Magento\Framework\Exception\LocalizedException(
                __('Could not connect to our payment servers. Please, try again later...')
            );
        }
    }

    /**
     * @param \Magento\Sales\Model\Order $order
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    private function requestMbway($order, $checkout_data)
    {
        $frequentConfig = $this->_easypayConfig->getFrequent();
        $orderInfo = $this->getInfoInstance();
        $body = [
            'key' => $order->getIncrementId(),
            'currency' => $order->getBaseCurrency()->getCode(),
            'max_value' => (float) $frequentConfig['max_value'],
            'min_value' => (float) $frequentConfig['min_value'],
            'unlimited_payments' => true,
        ];
        $billingInfo = $order->getBillingAddress();
        $body['customer'] = [
            'key' => $order->getCustomerId() ?? '',
            'name' => $order->getCustomerName() ?? '',
            'email' => $order->getCustomerEmail() ?? '',
            'phone' =>  $checkout_data['phone_number'],
            'phone_indicative' => '+351',
            'fiscal_number' => $order->getCustomerTaxvat() ?? '',
        ];
        $response = $this->_easypayApi->createFrequentPayment(
            'mbw',
            $body
        );
        if ($response) {
            if (isset($response['status']) && $response['status'] === 'ok') {
                $orderInfo->setAdditionalInformation('easypay', $response);
                /** @var \Toogas\Easypay\Model\EasypayPayment $payment */
                $payment = $this->_easypayPaymentFactory->create();
                $payment->setData([
                    'order_id' => $order->getIncrementId(),
                    'payment_id' => $response['id'],
                    'status' => 'created',
                    'type' => 'frequent_create',
                    'method' => self::EASYPAY_METHOD . '-mbw'
                ]);
                $payment->save();
                /** @var \Toogas\Easypay\Model\EasypayApiTransaction $transaction */
                $transaction = $this->_easypayApiTransactionFactory->create();
                $transaction->setData([
                    'payment_id' => $response['id'],
                    'origin' => 'magento',
                    'api_transaction_json' => json_encode($response, JSON_PRETTY_PRINT)
                ]);
                $transaction->save();
            } else if(isset($response['status']) && $response['status'] === 'error') {
                throw new \Magento\Framework\Exception\LocalizedException(
                    __($response['message'][0])
                );
            } else {
                throw new \Magento\Framework\Exception\LocalizedException(
                    __('Could not process your payment.')
                );
            }
        } else {
            throw new \Magento\Framework\Exception\LocalizedException(
                __('Could not connect to our payment servers. Please, try again later...')
            );
        }
    }

    /**
     * @param \Magento\Sales\Model\Order $order
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    private function requestCreditCard($order)
    {
        throw new \Magento\Framework\Exception\LocalizedException(
            __('This method is not available at the moment.')
        );
        return;
        $frequentConfig = $this->_easypayConfig->getFrequent();
        $orderInfo = $this->getInfoInstance();
        $body = [
            'key' => $order->getIncrementId(),
            'currency' => $order->getBaseCurrency()->getCode(),
            'max_value' => (float) $frequentConfig['max_value'],
            'min_value' => (float) $frequentConfig['min_value'],
            'unlimited_payments' => true,
        ];
        $billingInfo = $order->getBillingAddress();
        $body['customer'] = [
            'key' => $order->getCustomerId() ?? '',
            'name' => $order->getCustomerName() ?? '',
            'email' => $order->getCustomerEmail() ?? '',
            'phone' => $billingInfo->getTelephone() ?? '',
            'fiscal_number' => $order->getCustomerTaxvat() ?? '',
        ];
        $response = $this->_easypayApi->createFrequentPayment(
            'cc',
            $body
        );
        if ($response) {
            if (isset($response['status']) && $response['status'] === 'ok' && isset($response['method']['url']) ) {
                $orderInfo->setAdditionalInformation('easypay', $response);
                /** @var \Toogas\Easypay\Model\EasypayPayment $payment */
                $payment = $this->_easypayPaymentFactory->create();
                $payment->setData([
                    'order_id'      => $order->getIncrementId(),
                    'payment_id'    => $response['id'],
                    'status'        => 'created',
                    'type'          => 'frequent_create',
                    'method'        => self::EASYPAY_METHOD . '-cc'
                ]);
                $payment->save();
                /** @var \Toogas\Easypay\Model\EasypayApiTransaction $transaction */
                $transaction = $this->_easypayApiTransactionFactory->create();
                $transaction->setData([
                    'payment_id'            => $response['id'],
                    'origin'                => 'magento',
                    'api_transaction_json'  => json_encode($response, JSON_PRETTY_PRINT)
                ]);
                $transaction->save();
            } else {
                throw new \Magento\Framework\Exception\LocalizedException(
                    __('Could not process your payment.')
                );
            }
        } else {
            throw new \Magento\Framework\Exception\LocalizedException(
                __('Could not connect to our payment servers. Please, try again later...')
            );
        }
    }

    /**
     * @param \Magento\Sales\Model\Order $order
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    private function requestDirectDebit($order, $checkout_data)
    {
        $frequentConfig = $this->_easypayConfig->getFrequent();
        $orderInfo = $this->getInfoInstance();
        $body = [
            'key' => $order->getIncrementId(),
            'currency' => $order->getBaseCurrency()->getCode(),
            'max_value' => (float) $frequentConfig['max_value'],
            'min_value' => (float) $frequentConfig['min_value'],
            'unlimited_payments' => true,
        ];
        $billingInfo = $order->getBillingAddress();
        $body['customer'] = [
            'key' => $order->getCustomerId() ?? '',
            'name' => $order->getCustomerName() ?? '',
            'email' => $order->getCustomerEmail() ?? '',
            'phone' => $billingInfo->getTelephone() ?? '',
            'fiscal_number' => $order->getCustomerTaxvat() ?? '',
        ];
        $body['sdd_mandate']['iban'] = $checkout_data['iban'];
        //$body['sdd_mandate']['key'] = $order->getCustomerId() ?? '';
        $body['sdd_mandate']['name'] = $order->getCustomerName() ?? '';
        $body['sdd_mandate']['email'] = $order->getCustomerEmail() ?? '';
        $body['sdd_mandate']['phone'] = $checkout_data['phone_number'];
        $body['sdd_mandate']['account_holder'] = $checkout_data['account_holder'];
        $body['sdd_mandate']['country_code'] = 'PT';
        //$body['sdd_mandate']['max_num_debits'] = '12';

        $response = $this->_easypayApi->createFrequentPayment(
            'dd',
            $body
        );

        if ($response) {
            if (isset($response['status']) && $response['status'] === 'ok') {
                $orderInfo->setAdditionalInformation('easypay', $response);
                /** @var \Toogas\Easypay\Model\EasypayPayment $payment */
                $payment = $this->_easypayPaymentFactory->create();
                $payment->setData([
                    'order_id'      => $order->getIncrementId(),
                    'payment_id'    => $response['id'],
                    'status'        => 'created',
                    'type'          => 'frequent_create',
                    'method'        => self::EASYPAY_METHOD . '-dd'
                ]);
                $payment->save();
                /** @var \Toogas\Easypay\Model\EasypayApiTransaction $transaction */
                $transaction = $this->_easypayApiTransactionFactory->create();
                $transaction->setData([
                    'payment_id'            => $response['id'],
                    'origin'                => 'magento',
                    'api_transaction_json'  => json_encode($response, JSON_PRETTY_PRINT)
                ]);
                $transaction->save();
            } else {
                throw new \Magento\Framework\Exception\LocalizedException(
                    __('Could not process your payment.')
                );
            }
        } else {
            throw new \Magento\Framework\Exception\LocalizedException(
                __('Could not connect to our payment servers. Please, try again later...')
            );
        }
    }

    /**
     * @return string
     */
    public function getConfigPaymentAction()
    {
        return self::ACTION_AUTHORIZE;
    }
}
