<?php

namespace Toogas\Easypay\Model\Payment;

use Magento\Directory\Helper\Data as DirectoryHelper;
use Toogas\Easypay\Model\EasypayApiTransaction;
use Toogas\Easypay\Model\EasypayPayment;

class EasypayMbway extends \Magento\Payment\Model\Method\AbstractMethod
{
    const EASYPAY_METHOD = 'mbw';

    /** @var \Toogas\Easypay\Model\Http\EasypayApi */
    protected $_easypayApi;

    /** @var \Toogas\Easypay\Model\Config\EasypayConfig */
    protected $_easypayConfig;

    /** @var \Toogas\Easypay\Model\EasypayPaymentFactory */
    protected $_easypayPaymentFactory;

    /** @var \Toogas\Easypay\Model\EasypayApiTransactionFactory */
    protected $_easypayApiTransactionFactory;

    /** @var \Magento\Checkout\Model\Session */
    protected $_checkoutSession;

    protected $_infoBlockType = \Toogas\Easypay\Block\Adminhtml\Payment\Info::class;

    protected $_code = 'easypay_mbway';
    protected $_isInitializeNeeded = true;

    /**
     * EasypayMbway constructor.
     * @param \Magento\Framework\Model\Context $context
     * @param \Magento\Framework\Registry $registry
     * @param \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory
     * @param \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory
     * @param \Magento\Payment\Helper\Data $paymentData
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Payment\Model\Method\Logger $logger
     * @param \Toogas\Easypay\Model\Http\EasypayApi $easypayApi
     * @param \Toogas\Easypay\Model\Config\EasypayConfig $easypayConfig
     * @param \Toogas\Easypay\Model\EasypayPaymentFactory $easypayPaymentFactory
     * @param \Toogas\Easypay\Model\EasypayApiTransactionFactory $easypayApiTransactionFactory
     * @param \Magento\Checkout\Model\Session $checkoutSession
     * @param \Magento\Framework\Model\ResourceModel\AbstractResource|null $resource
     * @param \Magento\Framework\Data\Collection\AbstractDb|null $resourceCollection
     * @param array $data
     * @param DirectoryHelper|null $directory
     */
    public function __construct(
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory,
        \Magento\Payment\Helper\Data $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Payment\Model\Method\Logger $logger,
        \Toogas\Easypay\Model\Http\EasypayApi $easypayApi,
        \Toogas\Easypay\Model\Config\EasypayConfig $easypayConfig,
        \Toogas\Easypay\Model\EasypayPaymentFactory $easypayPaymentFactory,
        \Toogas\Easypay\Model\EasypayApiTransactionFactory $easypayApiTransactionFactory,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
        array $data = [],
        DirectoryHelper $directory = null
    ) {
        parent::__construct(
            $context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $paymentData,
            $scopeConfig,
            $logger,
            $resource,
            $resourceCollection,
            $data,
            $directory
        );
        $this->_easypayApi = $easypayApi;
        $this->_easypayConfig = $easypayConfig;
        $this->_easypayPaymentFactory = $easypayPaymentFactory;
        $this->_easypayApiTransactionFactory = $easypayApiTransactionFactory;
        $this->_checkoutSession = $checkoutSession;
    }

    public function isAvailable(\Magento\Quote\Api\Data\CartInterface $quote = null)
    {
        $isAvailable = parent::isAvailable($quote);
        return ($this->_easypayConfig->getFrequent()['active'] !== '1') && $isAvailable;
    }

    public function validate()
    {
        $info = $this->getInfoInstance()->getAdditionalInformation('checkout_data');
        if (!isset($info['phone_number'])          ||
            $info['phone_number'] === ''           ||
            !ctype_digit($info['phone_number'])    ||
            strlen($info['phone_number']) !== 9) {
            throw new \Magento\Framework\Exception\LocalizedException(
                __('You need to input a valid phone a number')
            );
        }
        return parent::validate();
    }

    /**
     * @param string $paymentAction
     * @param object $stateObject
     * @return \Magento\Payment\Model\Method\AbstractMethod|EasypayMbway
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function initialize($paymentAction, $stateObject)
    {
        $state = \Magento\Sales\Model\Order::STATE_NEW;
        $status = 'pending';
        $stateObject->setState($state);
        $stateObject->setStatus($status);
        $stateObject->setIsNotified(false);
        $order = $this->getInfoInstance()->getOrder();
        return $this->requestMbw($order);
    }

    /**
     * @param \Magento\Sales\Model\Order $order
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function requestMbw($order)
    {
        $orderInfo = $this->getInfoInstance();
        $body = [
            'key' => $order->getIncrementId()
        ];
        $body['customer'] = [
            //'key' => $order->getCustomerId() ?: '',
            'name' => $order->getCustomerName() ?: '',
            'email' => $order->getCustomerEmail() ?: '',
            'phone' => $orderInfo->getAdditionalInformation('checkout_data')['phone_number'] ?? '',
            'fiscal_number' => $order->getCustomerTaxvat() ?: '',
        ];
        $response = $this->_easypayApi->createSinglePayment(
            self::EASYPAY_METHOD,
            (float) $order->getGrandTotal(),
            $this->getPaymentType(),
            'Pagamento de Encomenda',
            $body
        );
        if ($response) {
            if (isset($response['status']) && $response['status'] === 'ok') {
                $this->getInfoInstance()->setAdditionalInformation('easypay', $response);
                /** @var \Toogas\Easypay\Model\EasypayPayment $payment */
                $payment = $this->_easypayPaymentFactory->create();
                $payment->setData([
                    'order_id'      => $order->getIncrementId(),
                    'payment_id'    => $response['id'],
                    'status'        => 'created',
                    'type'          => $this->getPaymentType(),
                    'method'        => self::EASYPAY_METHOD
                ]);
                $payment->save();
                /** @var \Toogas\Easypay\Model\EasypayApiTransaction $transaction */
                $transaction = $this->_easypayApiTransactionFactory->create();
                $transaction->setData([
                    'payment_id'            => $response['id'],
                    'origin'                => 'magento',
                    'api_transaction_json'  => json_encode($response, JSON_PRETTY_PRINT)
                ]);
                $transaction->save();
            } else {
                throw new \Magento\Framework\Exception\LocalizedException(
                    __('Could not process your payment. Please make sure you have MBWay associated with your phone number.')
                );
            }
        } else {
            throw new \Magento\Framework\Exception\LocalizedException(
                __('Could not connect to our payment servers. Please, try again later...')
            );
        }
        return $this;
    }

    protected function getPaymentType()
    {
        return $this->_easypayConfig->getMbway()['payment_type'];
    }

    public function getConfigPaymentAction()
    {
        return self::ACTION_AUTHORIZE;
    }
}
