<?php

namespace Toogas\Easypay\Model\Payment;

use Magento\Directory\Helper\Data as DirectoryHelper;

class EasypayMultibanco extends \Magento\Payment\Model\Method\AbstractMethod
{
    const EASYPAY_METHOD = 'mb';

    /** @var \Toogas\Easypay\Model\Http\EasypayApi */
    protected $_easypayApi;

    /** @var \Toogas\Easypay\Model\Config\EasypayConfig */
    protected $_easypayConfig;

    /** @var \Toogas\Easypay\Model\EasypayPaymentFactory */
    protected $_easypayPaymentFactory;

    /** @var \Toogas\Easypay\Model\EasypayApiTransactionFactory */
    protected $_easypayApiTransactionFactory;

    /** @var \Magento\Checkout\Model\Session */
    protected $_checkoutSession;

    /** @var \Toogas\Easypay\Block\Adminhtml\Payment\Info */
    protected $_infoBlockType = \Toogas\Easypay\Block\Adminhtml\Payment\Info::class;

    protected $_code = 'easypay_multibanco';
    protected $_isInitializeNeeded = true;

    /**
     * EasypayMultibanco constructor.
     * @param \Magento\Framework\Model\Context $context
     * @param \Magento\Framework\Registry $registry
     * @param \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory
     * @param \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory
     * @param \Magento\Payment\Helper\Data $paymentData
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Payment\Model\Method\Logger $logger
     * @param \Toogas\Easypay\Model\Http\EasypayApi $easypayApi
     * @param \Magento\Checkout\Model\Session $checkoutSession
     * @param \Toogas\Easypay\Model\Config\EasypayConfig $easypayConfig
     * @param \Toogas\Easypay\Model\EasypayPaymentFactory $easypayPaymentFactory
     * @param \Toogas\Easypay\Model\EasypayApiTransactionFactory $easypayApiTransactionFactory
     * @param \Magento\Framework\Model\ResourceModel\AbstractResource|null $resource
     * @param \Magento\Framework\Data\Collection\AbstractDb|null $resourceCollection
     * @param array $data
     * @param DirectoryHelper|null $directory
     */
    public function __construct(
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory,
        \Magento\Payment\Helper\Data $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Payment\Model\Method\Logger $logger,
        \Toogas\Easypay\Model\Http\EasypayApi $easypayApi,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Toogas\Easypay\Model\Config\EasypayConfig $easypayConfig,
        \Toogas\Easypay\Model\EasypayPaymentFactory $easypayPaymentFactory,
        \Toogas\Easypay\Model\EasypayApiTransactionFactory $easypayApiTransactionFactory,
        \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
        array $data = [],
        DirectoryHelper $directory = null
    ) {
        parent::__construct(
            $context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $paymentData,
            $scopeConfig,
            $logger,
            $resource,
            $resourceCollection,
            $data,
            $directory
        );
        $this->_easypayApi = $easypayApi;
        $this->_easypayConfig = $easypayConfig;
        $this->_easypayPaymentFactory = $easypayPaymentFactory;
        $this->_easypayApiTransactionFactory = $easypayApiTransactionFactory;
        $this->_checkoutSession = $checkoutSession;
    }

    public function isAvailable(\Magento\Quote\Api\Data\CartInterface $quote = null)
    {
        $isAvailable = parent::isAvailable($quote);
        return ($this->_easypayConfig->getFrequent()['active'] !== '1') && $isAvailable;
    }

    public function initialize($paymentAction, $stateObject)
    {
        $state = \Magento\Sales\Model\Order::STATE_NEW;
        $status = 'pending';
        $stateObject->setState($state);
        $stateObject->setStatus($status);
        $stateObject->setIsNotified(false);
        $order = $this->getInfoInstance()->getOrder();
        $this->requestMb($order);
        return $this;
    }

    /**
     * @param \Magento\Sales\Model\Order $order
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function requestMb($order)
    {
        $orderInfo = $this->getInfoInstance();
        $multibancoConfig = $this->_easypayConfig->getMultibanco();
        $valid_days = 7;
        if (isset($multibancoConfig['valid_days']) && $multibancoConfig['valid_days'] > 0) {
            $valid_days = $multibancoConfig['valid_days'];
        }
        $expiration_time = date('Y-m-d H:i', strtotime('+' . $valid_days . ' days'));
        $body = [
            'key' => $order->getIncrementId(),
            'expiration_time' => $expiration_time
        ];
        $body['customer'] = [
            //'key' => $order->getCustomerId() ?: '',
            'name' => $order->getCustomerName() ?: '',
            'email' => $order->getCustomerEmail() ?: '',
            'fiscal_number' => $order->getCustomerTaxvat() ?: '',
        ];
        $response = $this->_easypayApi->createSinglePayment(
            self::EASYPAY_METHOD,
            (float) $order->getGrandTotal(),
            'sale',
            'Pagamento de Encomenda',
            $body
        );
        if ($response) {
            if (isset($response['status']) && $response['status'] === 'ok') {
                $orderInfo->setAdditionalInformation('easypay', $response);
                $orderInfo->setAdditionalInformation('checkout_data', ['expiration_time' => $expiration_time]);
                /** @var \Toogas\Easypay\Model\EasypayPayment $payment */
                $payment = $this->_easypayPaymentFactory->create();
                $payment->setData([
                    'order_id'      => $order->getIncrementId(),
                    'payment_id'    => $response['id'],
                    'status'        => 'created',
                    'type'          => 'sale',
                    'method'        => self::EASYPAY_METHOD
                ]);
                $payment->save();
                /** @var \Toogas\Easypay\Model\EasypayApiTransaction $transaction */
                $transaction = $this->_easypayApiTransactionFactory->create();
                $transaction->setData([
                    'payment_id'            => $response['id'],
                    'origin'                => 'magento',
                    'api_transaction_json'  => json_encode($response, JSON_PRETTY_PRINT)
                ]);
                $transaction->save();
            } else {
                throw new \Magento\Framework\Exception\LocalizedException(
                    __('Could not generate a reference.')
                );
            }
        } else {
            throw new \Magento\Framework\Exception\LocalizedException(
                __('Could not connect to our payment servers. Please, try again later...')
            );
        }
        return $this;
    }

    public function getConfigPaymentAction()
    {
        return self::ACTION_AUTHORIZE;
    }
}
